export default class RealisticTemperatureSensor {
    constructor({
      interval = 1000,
      base = 15,
      amplitude = 10,
      arCoeff = 0.8,
      noiseSigma = 0.5,
      spikeChance = 0.01,
      spikeSize = 10,
    } = {}) {
      this.interval = interval;
      this.base = base;
      this.amplitude = amplitude;
      this.arCoeff = arCoeff;
      this.noiseSigma = noiseSigma;
      this.spikeChance = spikeChance;
      this.spikeSize = spikeSize;

      this.anomaly = 0; // AR(1) state
      this.t = 0;
      this.timer = null;
    }

    // Seasonal cycle, like Juneau model (can swap with real data later)
    seasonalMean(t) {
      const dayOfYear = t % (365 * 40) ;
      const radians = (2 * Math.PI * dayOfYear) / (365 * 40);
      return this.base + this.amplitude * Math.sin(radians); // e.g., 15 ±10°C
    }

    // Main update step
    generate() {
      const seasonal = this.seasonalMean(this.t);
      const noise = this.noiseSigma * (Math.random() * 2 - 1);
      this.anomaly = this.arCoeff * this.anomaly + noise;

      // Occasional fault spike
      let spike = 0;
      if (Math.random() < this.spikeChance) {
        spike = this.spikeSize * (0.5 + Math.random());
      }

      const temp = seasonal + this.anomaly + spike;
      return {
        temperature: Math.min(Math.max(parseFloat(temp.toFixed(2)), 0), 50),
        time: this.t,
      };
    }

    start(onData) {
      if (this.timer) return;
      this.timer = setInterval(() => {
        this.t += 1;
        onData(this.generate());
      }, this.interval);
    }

    stop() {
      clearInterval(this.timer);
      this.timer = null;
    }
  }

