"use strict";
var __awaiter = (this && this.__awaiter) || function (thisArg, _arguments, P, generator) {
    function adopt(value) { return value instanceof P ? value : new P(function (resolve) { resolve(value); }); }
    return new (P || (P = Promise))(function (resolve, reject) {
        function fulfilled(value) { try { step(generator.next(value)); } catch (e) { reject(e); } }
        function rejected(value) { try { step(generator["throw"](value)); } catch (e) { reject(e); } }
        function step(result) { result.done ? resolve(result.value) : adopt(result.value).then(fulfilled, rejected); }
        step((generator = generator.apply(thisArg, _arguments || [])).next());
    });
};
Object.defineProperty(exports, "__esModule", { value: true });
const mqtt_1 = require("mqtt");
const config_1 = require("./config");
/**
 * MQTT Client Singleton
 * This class implements the Singleton pattern to ensure only one instance exists.
 * It provides methods to subscribe to topics, add message listeners, and close the connection.
 * @class MQTT
 * @singleton
 * @example
 * const mqtt_instance = MQTT.getInstance();
 * mqtt_instance.Subscribe("your/topic");
 * mqtt_instance.AddOnMessageListener((topic, message) => {
 *   console.log(`Received message on topic ${topic}:`, message.toString());
 * });
 * mqtt_instance.close(); // To close the connection when done
 */
class MQTT {
    constructor() {
        if (MQTT.instance) {
            return MQTT.instance;
        }
        this.client = (0, mqtt_1.connect)(config_1.mqttBrokerUrl);
        MQTT.instance = this;
    }
    /**
     * Returns the singleton MQTT instance, creating it if necessary.
     * @return {MQTT} The singleton MQTT instance.
     */
    static getInstance() {
        if (!MQTT.instance) {
            MQTT.instance = new MQTT();
        }
        return MQTT.instance;
    }
    /**
     * Subscribes to a given MQTT topic.
     * @param {string} topic - The topic to subscribe to.
     * @returns {Promise<boolean>} A promise that resolves to true if the subscription was successful, false otherwise.
     */
    Subscribe(topic) {
        return __awaiter(this, void 0, void 0, function* () {
            return new Promise((resolve, reject) => {
                var _a;
                (_a = this.client) === null || _a === void 0 ? void 0 : _a.subscribe(topic, (err) => {
                    if (err) {
                        console.error("MQTT: Subscription attempt error:", err);
                        reject(err);
                    }
                    else {
                        console.log(`MQTT: Successfully Subscribed to topic: ${topic}`);
                        resolve(true);
                    }
                });
            });
        });
    }
    /**
     * Adds a listener for incoming MQTT messages.
     * @param {OnMessageCallback} callback
     */
    AddOnMessageListener(callback) {
        var _a;
        (_a = this.client) === null || _a === void 0 ? void 0 : _a.on("message", callback);
    }
    /**
     * Closes the MQTT client connection.
     * @returns {void}
     */
    close() {
        var _a;
        (_a = this.client) === null || _a === void 0 ? void 0 : _a.end(() => {
            console.log("MQTT client disconnected");
        });
    }
}
MQTT.instance = null;
exports.default = MQTT;
