"use strict";
var __awaiter = (this && this.__awaiter) || function (thisArg, _arguments, P, generator) {
    function adopt(value) { return value instanceof P ? value : new P(function (resolve) { resolve(value); }); }
    return new (P || (P = Promise))(function (resolve, reject) {
        function fulfilled(value) { try { step(generator.next(value)); } catch (e) { reject(e); } }
        function rejected(value) { try { step(generator["throw"](value)); } catch (e) { reject(e); } }
        function step(result) { result.done ? resolve(result.value) : adopt(result.value).then(fulfilled, rejected); }
        step((generator = generator.apply(thisArg, _arguments || [])).next());
    });
};
Object.defineProperty(exports, "__esModule", { value: true });
const pg_1 = require("pg");
const { TIMESERIES_DATABASE_URL } = process.env;
/**
 * Database class for managing PostgreSQL connections.
 * This class implements the Singleton pattern to ensure only one instance exists.
 * It provides methods to execute SQL queries and manage the connection pool.
 * * @class Database
 * @singleton
 * @example
 * const db = Database.getInstance();
 * db.query('SELECT * FROM table_name', [])
 */
class Database {
    constructor() {
        if (Database.instance) {
            return Database.instance;
        }
        if (!TIMESERIES_DATABASE_URL) {
            throw new Error("Database configuration environment variables are not set");
        }
        this.pool = new pg_1.Pool({
            connectionString: TIMESERIES_DATABASE_URL,
        });
        Database.instance = this;
    }
    /**
     * Returns the singleton Database instance, creating it if necessary.
     * @return {Database} The singleton Database instance.
     */
    static getInstance() {
        if (!Database.instance) {
            Database.instance = new Database();
        }
        return Database.instance;
    }
    /**
     * Executes a SQL query with parameters.
     * @param {string} sql - The SQL query string.
     * @param {any[]} params - The parameters for the SQL query.
     * @returns {Promise<QueryResult>} The query result promise.
     * @throws {Error} If the database pool is not initialized.
     */
    query(sql, params) {
        return __awaiter(this, void 0, void 0, function* () {
            if (!this.pool) {
                throw new Error("Database pool is not initialized");
            }
            return this.pool.query(sql, params);
        });
    }
    /**
     * Closes the database connection pool and resets the singleton instance.
     * @returns {Promise<void>} A promise that resolves when the pool is closed.
     */
    close() {
        return __awaiter(this, void 0, void 0, function* () {
            var _a;
            yield ((_a = this.pool) === null || _a === void 0 ? void 0 : _a.end());
            Database.instance = null;
        });
    }
}
Database.instance = null;
exports.default = Database;
