    import { Test, TestingModule } from '@nestjs/testing';
import { INestApplication, ValidationPipe } from '@nestjs/common';
import * as request from 'supertest';
import { AppModule } from '../src/app.module';
import { ALLOWED_ENDPOINTS_SENSOR } from '../src/fuzzy-core/fuzzy.types';

describe('MQTT Endpoints (e2e)', () => {
  let app: INestApplication;
  const machineId = 1;
  const sensor = ALLOWED_ENDPOINTS_SENSOR[0]; // use the first allowed sensor
  const topic = `${machineId}/${sensor}`;

  beforeAll(async () => {
    process.env.MQTT_HOST_IP = 'localhost'; // Adjust for test broker or mock

    const moduleFixture: TestingModule = await Test.createTestingModule({
      imports: [AppModule],
    }).compile();

    app = moduleFixture.createNestApplication();
    app.useGlobalPipes(new ValidationPipe());
    await app.init();
  });

  afterAll(async () => {
    await app.close();
  });

  it('/mqtt/client/status (GET)', async () => {
    const res = await request(app.getHttpServer()).get('/mqtt/client/status');
    expect(res.status).toBe(200);
    expect(res.text).toMatch(/connected|not connected/i);
  });

  it('/mqtt/client/connect (GET)', async () => {
    const res = await request(app.getHttpServer()).get('/mqtt/client/connect');
    expect(res.status).toBe(200);
    expect(res.text).toContain('Connected');
  });

  it('/mqtt/client/disconnect (DELETE)', async () => {
    const res = await request(app.getHttpServer()).delete('/mqtt/client/disconnect');
    expect(res.status).toBe(200);
    expect(res.text).toContain('Disconnected');
  });

  it('/mqtt/client/connect (GET again)', async () => {
    const res = await request(app.getHttpServer()).get('/mqtt/client/connect');
    expect(res.status).toBe(200);
  });

  it('/mqtt/subscriptions/status (GET)', async () => {
    const res = await request(app.getHttpServer()).get('/mqtt/subscriptions/status');
    expect(res.status).toBe(200);
  });

  it(`/mqtt/latest/${machineId}/${sensor} (GET)`, async () => {
    const res = await request(app.getHttpServer()).get(`/mqtt/latest/${machineId}/${sensor}`);
    expect(res.status).toBe(200);
    expect(res.text).toBeDefined();
  });

  it(`/mqtt/debug/${machineId}/${sensor} (GET HTML)`, async () => {
    const res = await request(app.getHttpServer()).get(`/mqtt/debug/${machineId}/${sensor}`);
    expect(res.status).toBe(200);
    expect(res.text).toContain('<html');
  });

  it(`/mqtt/${machineId}/${sensor} (DELETE unsubscribe)`, async () => {
    const res = await request(app.getHttpServer()).delete(`/mqtt/${machineId}/${sensor}`);
    expect(res.status).toBe(200);
    expect(res.text).toContain(`Unsubscribed`);
  });

  it(`/mqtt/all (DELETE unsubscribe all)`, async () => {
    const res = await request(app.getHttpServer()).delete(`/mqtt/all`);
    expect(res.status).toBe(200);
    expect(res.text).toContain('Unsubscribed');
  });
});
