import { Injectable } from '@nestjs/common';
import { Graph, GraphNode } from './navigation.entity';

@Injectable()
export class PathfindService {
  constructor() {}

  /**
   * Find a path between two nodes using the A* algorithm.
   * @param graph The graph containing the nodes and edges.
   * @param from The starting node.
   * @param to The destination node.
   * @returns An array of GraphNode representing the path from 'from' to 'to'.
   */
  AStar(graph: Graph, from: GraphNode, to: GraphNode): GraphNode[] {
    // Helper to get node by id
    const nodeMap = new Map<string, GraphNode>(
      graph.nodes.map((n) => [n.id, n]),
    );
    // Heuristic: Euclidean distance in 3D
    const heuristic = (a: GraphNode, b: GraphNode) => {
      const dx = a.x - b.x;
      const dy = a.y - b.y;
      const dz = a.z - b.z;
      return Math.sqrt(dx * dx + dy * dy + dz * dz);
    };

    const openSet: Set<string> = new Set([from.id]);
    const cameFrom: Record<string, string | null> = {};
    const gScore: Record<string, number> = {};
    const fScore: Record<string, number> = {};

    for (const node of graph.nodes) {
      gScore[node.id] = Infinity;
      fScore[node.id] = Infinity;
      cameFrom[node.id] = null;
    }
    gScore[from.id] = 0;
    fScore[from.id] = heuristic(from, to);

    while (openSet.size > 0) {
      // Get node in openSet with lowest fScore
      let currentId: string | null = null;
      let minF = Infinity;
      for (const id of openSet) {
        if (fScore[id] < minF) {
          minF = fScore[id];
          currentId = id;
        }
      }
      if (!currentId) break;
      if (currentId === to.id) {
        // Reconstruct path
        const path: GraphNode[] = [];
        let curr: string | null = to.id;
        while (curr) {
          path.unshift(nodeMap.get(curr)!);
          curr = cameFrom[curr];
        }
        return path;
      }
      openSet.delete(currentId);
      const current = nodeMap.get(currentId)!;
      for (const edge of current.edges) {
        const neighborId =
          edge.nodeAId === current.id ? edge.nodeBId : edge.nodeAId;

        const neighbor = nodeMap.get(neighborId);
        if (!neighbor) continue;
        const tentativeG = gScore[current.id] + heuristic(current, neighbor);
        if (tentativeG < gScore[neighbor.id]) {
          cameFrom[neighbor.id] = current.id;
          gScore[neighbor.id] = tentativeG;
          fScore[neighbor.id] = tentativeG + heuristic(neighbor, to);
          openSet.add(neighbor.id);
        }
      }
    }
    // No path found
    return [];
  }
}
