import { Controller, Delete, Param, Req, Sse, Get } from '@nestjs/common';
import { IncomingMessage } from 'http';
import { MqttService } from './mqtt.service';
import { Observable } from 'rxjs';
import { ParseSensorPipe } from './parse-sensor.pipe';
import {
  ALLOWED_ENDPOINTS_SENSOR,
  SensorName,
} from '../fuzzy-core/fuzzy.types';
import {
  ApiParam,
  ApiTags,
  ApiProduces,
  ApiResponse,
  ApiOperation,
} from '@nestjs/swagger';

/**
 * Controller to handle MQTT-related requests.
 * @class MqttSubscriptionsController
 */
@Controller('mqtt')
@ApiTags('MQTT Subscription Management')
export class MqttSubscriptionsController {
  constructor(private readonly mqttService: MqttService) {}

  /**
   * Endpoint to get the status of the MQTT subscriptions.
   * @return {string} List of subscribed topics.
   */
  @Get('subscriptions/status')
  @ApiOperation({ summary: 'Will get a list of all subscribed topics' })
  @ApiProduces('text/plain')
  @ApiResponse({ status: 200, description: 'List of subscribed topics' })
  getSubscriptionsStatus(): string {
    return this.mqttService.getSubscriptionsStatus();
  }

  /**
   * SSE endpoint to listen to MQTT messages for a specific machine ID.
   * @param machineId
   * @param sensor
   * @param request
   * @return {Observable<{ data: string }>} Observable of string messages.
   */
  @Sse(':machineId/:sensor')
  @ApiOperation({ summary: 'Live stream of sensor data via SSE' })
  @ApiProduces('text/event-stream')
  @ApiResponse({ status: 200, description: 'Stream of sensor events' })
  @ApiParam({
    name: 'sensor',
    enum: ALLOWED_ENDPOINTS_SENSOR,
    description: 'Sensor type for the machine',
  })
  getLiveUpdates(
    @Param('machineId') machineId: number,
    @Param('sensor', ParseSensorPipe) sensor: SensorName,
    @Req() request: IncomingMessage,
  ): Observable<{ data: string }> {
    return this.mqttService.getLiveUpdates(`${machineId}/${sensor}`, request);
  }

  /**
   * SSE endpoint to listen to all MQTT messages.
   * @return {Observable<{ data: string }>} Observable of string messages.
   */
  @Sse('all')
  @ApiOperation({ summary: 'Live stream of all sensor data via SSE' })
  @ApiProduces('text/event-stream')
  @ApiResponse({ status: 200, description: 'Stream of all sensor events' })
  getAllLiveUpdates(
    @Req() request: IncomingMessage,
  ): Observable<{ data: string }> {
    return this.mqttService.getLiveUpdates('#', request);
  }

  /**
   * Endpoint to unsubscribe to a specific machine ID.
   * @param machineId
   * @param sensor
   * @return {string} Unsubscription message.
   */
  @Delete(':machineId/:sensor')
  @ApiOperation({ summary: 'Unsubscribe from a specific machine ID' })
  @ApiProduces('text/plain')
  @ApiResponse({ status: 200, description: 'Unsubscribed from the machine' })
  @ApiParam({
    name: 'sensor',
    enum: ALLOWED_ENDPOINTS_SENSOR,
    description: 'Sensor type for the machine',
  })
  deleteMachine(
    @Param('machineId') machineId: number,
    @Param('sensor', ParseSensorPipe) sensor: SensorName,
  ): string {
    this.mqttService.stopListeningTo(`${machineId}/${sensor}`);
    return `Unsubscribed from ${machineId}`;
  }

  /**
   * Endpoint to unsubscribe to all machine IDs.
   * @return {string} Unsubscription message.
   */
  @Delete('all')
  @ApiOperation({ summary: 'Unsubscribe from all machine IDs' })
  @ApiProduces('text/plain')
  @ApiResponse({ status: 200, description: 'Unsubscribed from all machines' })
  deleteAllMachines(): string {
    this.mqttService.stopListeningTo('#');
    return `Unsubscribed from all machines`;
  }
}
