import { Controller, Get, Param } from '@nestjs/common';
import { MqttService } from './mqtt.service';
import {
  ApiTags,
  ApiOperation,
  ApiResponse,
  ApiProduces,
  ApiParam,
} from '@nestjs/swagger';
import { ParseSensorPipe } from './parse-sensor.pipe';
import {
  ALLOWED_ENDPOINTS_SENSOR,
  SensorName,
} from '../fuzzy-core/fuzzy.types';

/**
 * Controller to handle MQTT-related requests.
 * @class MqttDebugController
 */
@Controller('mqtt')
@ApiTags('MQTT Debug & Testing')
export class MqttDebugController {
  constructor(private readonly mqttService: MqttService) {}

  /**
   * Endpoint to get the latest message for a specific machine ID.
   * @description This is for testing purposes only. It returns the latest message.
   * @param machineId
   * @param sensor
   * @return {string} Latest message
   */
  @Get('latest/:machineId/:sensor')
  @ApiOperation({ summary: 'Get the latest message for a specific machine ID' })
  @ApiProduces('text/plain')
  @ApiResponse({ status: 200, description: 'Latest message' })
  @ApiParam({
    name: 'sensor',
    enum: ALLOWED_ENDPOINTS_SENSOR,
    description: 'Sensor type for the machine',
  })
  getLatestData(
    @Param('machineId') machineId: number,
    @Param('sensor', ParseSensorPipe) sensor: SensorName,
  ): string {
    return this.mqttService.getLatestMessage(machineId, sensor);
  }

  /**
   * Endpoint to get the live data for a specific machine ID via html.
   * @description This is for testing purposes only. It returns a simple HTML page.
   * @param machineId
   * @param sensor
   * @return {string} HTML page with live data
   */
  @Get('debug/:machineId/:sensor')
  @ApiOperation({ summary: 'Get live data for a specific machine ID in html.' })
  @ApiProduces('text/html')
  @ApiResponse({ status: 200, description: 'Live data in html' })
  @ApiParam({
    name: 'sensor',
    enum: ALLOWED_ENDPOINTS_SENSOR,
    description: 'Sensor type for the machine',
  })
  getIndex(
    @Param('machineId') machineId: number,
    @Param('sensor', ParseSensorPipe) sensor: SensorName,
  ): string {
    return `
      <html lang="">
        <body>
          <h1>Latest MQTT Data for Machine 1</h1>
          <pre id="data" style="color: black;">Waiting for data...</pre>
          <script>
            const machineId = ${machineId};
            const sensor = "${sensor}";
            const source = new EventSource('/mqtt/' + machineId + '/' + sensor);
            source.onmessage = function(event) {
              const pre = document.getElementById('data');
              pre.innerText = event.data;
              if (event.data.startsWith('[ERROR]')) {
                pre.style.color = 'red';
              } else if (event.data.startsWith('[WARN]')) {
                pre.style.color = 'orange';
              } else {
                pre.style.color = 'black';
              }
            };
          </script>
        </body>
      </html>
    `;
  }
}
