import { Injectable } from '@nestjs/common';
import * as fs from 'fs';
import * as path from 'path';

import { FuzzyEngine } from './fuzzy.engine';
import { createEngineFor } from './fuzzy.engine.factory';
import { FLV } from './FLV';
import {
  EvaluationResult,
  MachineConfig,
  SensorData,
  SensorName,
} from './fuzzy.types';
import { LeftShoulder, RightShoulder, Triangular } from './MembershipFunctions';

@Injectable()
export class FuzzyService {
  private engines: Record<string, FuzzyEngine> = {};

  private readonly flvTemplates: Partial<Record<SensorName, FLV>> = {
    temperature: this.createTemperatureFLV(),
    pressure: this.createPressureFLV(),
    vibration: this.createVibrationFLV(),
  };

  constructor() {
    this.loadEnginesFromConfig();
  }

  private loadEnginesFromConfig() {
    const configDir = path.join(process.cwd(), 'config', 'fuzzy');
    const files = fs
      .readdirSync(configDir)
      .filter((f) => f.endsWith('.config.json'));

    for (const file of files) {
      const { machineId, sensors, rules } = JSON.parse(
        fs.readFileSync(path.join(configDir, file), 'utf8'),
      ) as MachineConfig;

      const flvs = Object.fromEntries(
        sensors.map((sensor) => [sensor, this.flvTemplates[sensor]]),
      ) as Partial<Record<SensorName, FLV>>;

      this.engines[machineId] = createEngineFor(machineId, flvs, rules);
    }
  }

  evaluate(data: SensorData): EvaluationResult {
    const engine = this.engines[data.machineId];
    if (!engine) {
      throw new Error(
        `No fuzzy engine registered for machine: ${data.machineId}`,
      );
    }
    const { label, score } = engine.evaluate(data);
    return { machineId: data.machineId, priority: label, score };
  }

  private createTemperatureFLV(): FLV {
    const flv = new FLV(
      'Temperature',
      ['Low', 'Medium', 'High'],
      [0, 5, 10, 15, 20, 25, 30, 50],
    );
    flv.AddValuesForLabels(
      flv.GetXvalues(),
      ['Low', 'Medium', 'High'],
      [
        LeftShoulder(flv.GetXvalues(), 5, 15),
        Triangular(flv.GetXvalues(), 5, 15, 30),
        RightShoulder(flv.GetXvalues(), 15, 30),
      ],
    );
    return flv;
  }

  private createPressureFLV(): FLV {
    const flv = new FLV(
      'Pressure',
      ['Low', 'Medium', 'High'],
      [0, 5, 10, 15, 20, 25, 30, 50],
    );
    flv.AddValuesForLabels(
      flv.GetXvalues(),
      ['Low', 'Medium', 'High'],
      [
        LeftShoulder(flv.GetXvalues(), 5, 15),
        Triangular(flv.GetXvalues(), 5, 15, 30),
        RightShoulder(flv.GetXvalues(), 15, 30),
      ],
    );
    return flv;
  }

  private createVibrationFLV(): FLV {
    const flv = new FLV(
      'Vibration',
      ['Low', 'Medium', 'High'],
      [0, 5, 10, 15, 20, 25, 30, 50],
    );
    flv.AddValuesForLabels(
      flv.GetXvalues(),
      ['Low', 'Medium', 'High'],
      [
        LeftShoulder(flv.GetXvalues(), 5, 15),
        Triangular(flv.GetXvalues(), 5, 15, 30),
        RightShoulder(flv.GetXvalues(), 15, 30),
      ],
    );
    return flv;
  }
}
