export type MembershipFn = (xValues: number[], ...params: number[]) => number[];

/**
 * Generates a triangular membership number array for the given x values.
 * @param xValues The x values to calculate the membership function for
 * @param a The left point of the triangle that has a membership of 0
 * @param b The middle point of the triangle that has a membership of 1
 * @param c The right point of the triangle that has a membership of 0
 * @returns The mapped x values to the membership function
 * @example
 * const xValues = [0, 1, 2, 3, 4, 5];
 * const result = Triangular(xValues, 0, 2, 4);
 * // result = [0, 0.5, 1, 0.5, 0, 0]
 */
export const Triangular: MembershipFn = (xValues, a, b, c) => {
  return xValues.map((x) => {
    if (x <= a || x >= c) return 0; // defined 0 points a and c
    if (x === b) return 1; // defined 1 point b
    if (x > a && x < b) return (x - a) / (b - a); // defined 0 to 1 points a and b
    return (c - x) / (c - b); // defined 1 to 0 points b and c
  });
};

/**
 * Generates a Left shoulder membership number array for the given x values.
 * @param xValues The x values to calculate the membership function for
 * @param a The left point of the shoulder that has a membership of 1
 * @param b The right point of the shoulder that has a membership of 0
 * @returns The mapped x values to the membership function
 * @example
 * const xValues = [0, 1, 2, 3, 4, 5];
 * const result = LeftShoulder(xValues, 1, 3);
 * // result = [1, 1, 0.5, 0, 0, 0]
 */
export const LeftShoulder: MembershipFn = (xValues, a, b) => {
  return xValues.map((x) => {
    if (x <= a) return 1; // defined 1 point a
    if (x > a && x < b) return (b - x) / (b - a); // defined 1 to 0 points a and b
    return 0; // defined 0 point b
  });
};
/**
 * Generates a Right shoulder membership number array for the given x values.
 * @param xValues The x values to calculate the membership function for
 * @param a The left point of the shoulder that has a membership of 0
 * @param b The right point of the shoulder that has a membership of 1
 * @returns The mapped x values to the membership function
 * @example
 * const xValues = [0, 1, 2, 3, 4, 5];
 * const result = RightShoulder(xValues, 1, 3);
 * // result = [0, 0, 0.5, 1, 1, 1]
 */
export const RightShoulder: MembershipFn = (xValues, a, b) => {
  return xValues.map((x) => {
    if (x >= b) return 1; // defined 1 point b
    if (x > a && x < b) return (x - a) / (b - a); // defined 0 to 1 points a and b
    return 0; // defined 0 point a
  });
};

/**
 * Generates a trapezoidal membership number array for the given x values.
 * @param xValues The x values to calculate the membership function for
 * @param a The left point of the trapezoid that has a membership of 0
 * @param b The left point of the trapezoid that has a membership of 1
 * @param c The right point of the trapezoid that has a membership of 1
 * @param d The right point of the trapezoid that has a membership of 0
 * @returns The mapped x values to the membership function
 * @example
 * const xValues = [0, 1, 2, 3, 4, 5];
 * const result = Trapezoidal(xValues, 1, 2, 3, 4);
 * // result = [0, 0, 1, 1, 0, 0]
 *
 */
export const Trapezoidal: MembershipFn = (xValues, a, b, c, d) => {
  return xValues.map((x) => {
    if (x <= a || x >= d) return 0; // defined 0 points a and d
    if (x > a && x < b) return (x - a) / (b - a); // defined 0 to 1 points a and b
    if (x >= b && x <= c) return 1; // defined 1 points b and c
    return (d - x) / (d - c); // defined 1 to 0 points c and d
  });
};
