import FLV_Error from '../errors/FLV_Error';

/**
 * Fuzzy Logic Variable (FLV) class
 * This class represents a fuzzy logic variable with a set of labels and their corresponding values.
 * It allows for the addition of values, retrieval of degree of membership (DoM), and interpolation.
 */
export class FLV {
  // TODO: add check for DOM sum to equal 1 at ever xValue

  private name: string;
  private labels: string[];
  private data: Map<number, Record<string, number>>;

  /**
   * Creates a new FLV
   * @param {string} name - The name of the FLV
   * @param {string[]} labels - The labels of the FLV for example ['Low', 'Medium', 'High']
   * @param {number[]} xValues - The x values of the FLV
   * @example
   * const flv = new FLV(
   *   'Temperature',
   *   ['Low', 'Medium', 'High'],
   *   [0, 5, 10, 15, 20, 25, 30]
   * );
   * flv.AddValuesForLabels(
   *   [0, 5, 10, 15, 20, 25, 30], // flv.GetXvalues() is also possible
   *   ['Low', 'Medium', 'High'],
   *   [
   *     [1.0, 1.0, 0.5, 0.0, 0.0, 0.0, 0.0],
   *     [0.0, 0.0, 0.5, 1.0, 0.75, 0.25, 0.0],
   *     [0.0, 0.0, 0.0, 0.0, 0.25, 0.75, 1.0]
   *   ]
   *);
   */
  constructor(name: string, labels: string[], xValues: number[]) {
    this.name = name;
    this.labels = labels;
    this.data = new Map<number, Record<string, number>>();
    for (const x of xValues) {
      this.data.set(x, {});
      for (const label of labels) {
        this.data.get(x)![label] = 0;
      }
    }
  }

  /**
   * Gets the x values that are defined in the FLV
   * @returns {number[]} the x values of the FLV
   */
  GetXvalues(): number[] {
    return Array.from(this.data.keys());
  }

  /**
   * Gets the labels of the FLV for example ['Low', 'Medium', 'High']
   * @returns {string[]} the labels of the FLV
   */
  GetLabels(): string[] {
    return this.labels;
  }

  /**
   * Adds a value to the FLV based on the x value and label
   * @param x the x value to add the value to
   * @param label the label to add the value to
   * @param value to be added to the FLV
   */
  AddValue(x: number, label: string, value: number) {
    this.AddValues([x], label, [value]);
  }

  /**
   * Adds values to the FLV based on the x values and label
   * @param x the x values to add the values to
   * @param label the label to add the values to
   * @param values to be added to the FLV
   */
  AddValues(x: number[], label: string, values: number[]) {
    this.AddValuesForLabels(x, [label], [values]);
  }

  /**
   * Adds values to the FLV for the given Labels and x values
   * @param x the x values to add the values to
   * @param labels to add the values to
   * @param values to be added to the FLV
   */
  AddValuesForLabels(x: number[], labels: string[], values: number[][]) {
    for (let labelIndex = 0; labelIndex < labels.length; labelIndex++) {
      const label = labels[labelIndex];
      if (!this.labels.includes(label))
        throw new FLV_Error(`${label} is not in FLV`);

      for (let i = 0; i < x.length; i++) {
        if (!this.data.has(x[i]))
          // TODO: what if x is not inside of number[], but is withing the actual range? In this case 1 would be in range but is not in the number[].
          throw new FLV_Error(`${x[i]} is not in the scope of the FLV`);
        this.data.get(x[i])![label] = values[labelIndex][i];
      }
    }
  }

  /**
   * Gets the Degree of Membership (DoM) for the given x value
   * @param x the x value to get the DoM for
   * @returns {Record<string, number>} the DoM for the given x value
   */
  GetDoM(x: number): Record<string, number> {
    if (this.data.has(x)) return this.data.get(x)!;

    // interpolation when not defined in FLV

    const lowerX = this.GetXValueBefore(x);
    const upperX = this.GetXValueAfter(x);

    const partOfLower = (upperX - x) / (upperX - lowerX);
    const partOfUpper = (x - lowerX) / (upperX - lowerX);

    const lowerDoM = this.data.get(lowerX)!;
    const upperDoM = this.data.get(upperX)!;

    const interpolatedValues: Record<string, number> = {};
    for (const label of this.labels) {
      interpolatedValues[label] =
        lowerDoM[label] * partOfLower + upperDoM[label] * partOfUpper;
    }
    // TODO: add the new values to the FLV for future reference
    return interpolatedValues;
  }

  /**
   * Helper method to get the x value before the given x value
   * @param x the x value to get the x value before
   * @returns {number} the x value before the given x value
   * @throws {FLV_Error} if the x value is not in the scope of the FLV
   */
  private GetXValueBefore(x: number): number {
    const keys = Array.from(this.data.keys());
    if (x < keys[0] || x > keys[keys.length - 1])
      throw new FLV_Error(`${x} is not in the scope of the FLV`);

    for (let i = 1; i < keys.length; i++) {
      if (x < keys[i]) {
        return keys[i - 1];
      }
    }
    throw new FLV_Error(`Unknown error with finding the value before: ${x}`);
  }

  /**
   * Helper method to get the x value after the given x value
   * @param x the x value to get the x value after
   * @returns {number} the x value after the given x value
   * @throws {FLV_Error} if the x value is not in the scope of the FLV
   */
  private GetXValueAfter(x: number): number {
    const keys = Array.from(this.data.keys());
    if (x < keys[0] || x > keys[keys.length - 1])
      throw new FLV_Error(`${x} is not in the scope of the FLV`);

    for (let i = keys.length - 2; i >= 0; i--) {
      if (x > keys[i]) {
        return keys[i + 1];
      }
    }
    throw new FLV_Error(`Unknown error with finding the value after: ${x}`);
  }
}
