import { PrismaClient } from '@prisma/client';
import {
  GraphEdge,
  Graph,
  GraphNode,
} from 'src/navigation-core/navigation.entity';

const prisma = new PrismaClient();

const nodes: GraphNode[] = [
  // first floor
  { id: '0', x: -400, y: 0, z: 0, label: '', graphId: '1', edges: [] },
  { id: '1', x: -300, y: 0, z: -160, label: '', graphId: '1', edges: [] },
  { id: '2', x: -240, y: 0, z: -30, label: '', graphId: '1', edges: [] },
  { id: '3', x: -300, y: 0, z: 150, label: '', graphId: '1', edges: [] },
  { id: '4', x: -60, y: 0, z: -90, label: '', graphId: '1', edges: [] },
  { id: '5', x: -120, y: 0, z: 90, label: '', graphId: '1', edges: [] },
  { id: '6', x: 100, y: 0, z: -230, label: '', graphId: '1', edges: [] },
  { id: '7', x: 160, y: 0, z: 0, label: '', graphId: '1', edges: [] },
  { id: '8', x: 100, y: 0, z: 170, label: '', graphId: '1', edges: [] },
  { id: '9', x: 300, y: 0, z: -160, label: '', graphId: '1', edges: [] },
  { id: '10', x: 350, y: 0, z: 140, label: '', graphId: '1', edges: [] },

  // second floor
  { id: '11', x: -330, y: 50, z: 40, label: '', graphId: '1', edges: [] },
  { id: '12', x: -300, y: 50, z: -140, label: '', graphId: '1', edges: [] },
  { id: '13', x: -200, y: 50, z: 50, label: '', graphId: '1', edges: [] },
  { id: '14', x: -110, y: 50, z: -230, label: '', graphId: '1', edges: [] },
  { id: '15', x: -60, y: 50, z: -70, label: '', graphId: '1', edges: [] },
  { id: '16', x: 20, y: 50, z: 90, label: '', graphId: '1', edges: [] },
  { id: '17', x: 170, y: 50, z: -20, label: '', graphId: '1', edges: [] },
  { id: '18', x: 70, y: 50, z: -90, label: '', graphId: '1', edges: [] },
  { id: '19', x: 60, y: 50, z: -200, label: '', graphId: '1', edges: [] },
  { id: '20', x: 220, y: 50, z: -190, label: '', graphId: '1', edges: [] },
  { id: '21', x: 240, y: 50, z: -90, label: '', graphId: '1', edges: [] },
  { id: '22', x: 410, y: 50, z: 0, label: '', graphId: '1', edges: [] },

  // staircase left
  {
    id: '23',
    x: -500,
    y: 0,
    z: 0,
    label: 'Staircase',
    graphId: '1',
    edges: [],
  },
  {
    id: '24',
    x: -525,
    y: 25,
    z: 0,
    label: 'Staircase',
    graphId: '1',
    edges: [],
  },
  {
    id: '25',
    x: -500,
    y: 50,
    z: 0,
    label: 'Staircase',
    graphId: '1',
    edges: [],
  },

  // staircase right
  { id: '26', x: 500, y: 0, z: 0, label: 'Staircase', graphId: '1', edges: [] },
  {
    id: '27',
    x: 525,
    y: 25,
    z: 0,
    label: 'Staircase',
    graphId: '1',
    edges: [],
  },
  {
    id: '28',
    x: 500,
    y: 50,
    z: 0,
    label: 'Staircase',
    graphId: '1',
    edges: [],
  },
];

const edges: GraphEdge[] = [
  // floor 1
  { id: '1', nodeAId: '23', nodeBId: '0' }, // left staircase
  { id: '2', nodeAId: '0', nodeBId: '1' },
  { id: '3', nodeAId: '0', nodeBId: '2' },
  { id: '4', nodeAId: '0', nodeBId: '3' },
  { id: '5', nodeAId: '1', nodeBId: '6' },
  { id: '6', nodeAId: '2', nodeBId: '4' },
  { id: '7', nodeAId: '2', nodeBId: '5' },
  { id: '8', nodeAId: '3', nodeBId: '5' },
  { id: '9', nodeAId: '4', nodeBId: '7' },
  { id: '10', nodeAId: '4', nodeBId: '9' },
  { id: '11', nodeAId: '5', nodeBId: '7' },
  { id: '12', nodeAId: '5', nodeBId: '8' },
  { id: '13', nodeAId: '6', nodeBId: '9' },
  { id: '14', nodeAId: '7', nodeBId: '9' },
  { id: '15', nodeAId: '8', nodeBId: '10' },
  { id: '16', nodeAId: '9', nodeBId: '26' }, // right staircase
  { id: '17', nodeAId: '10', nodeBId: '26' }, // right staircase

  // floor 2
  { id: '18', nodeAId: '25', nodeBId: '11' }, // left staircase
  { id: '19', nodeAId: '25', nodeBId: '12' }, // left staircase
  { id: '20', nodeAId: '11', nodeBId: '13' },
  { id: '21', nodeAId: '12', nodeBId: '14' },
  { id: '22', nodeAId: '13', nodeBId: '15' },
  { id: '23', nodeAId: '14', nodeBId: '15' },
  { id: '24', nodeAId: '14', nodeBId: '19' },
  { id: '25', nodeAId: '15', nodeBId: '16' },
  { id: '26', nodeAId: '16', nodeBId: '17' },
  { id: '27', nodeAId: '16', nodeBId: '22' },
  { id: '28', nodeAId: '17', nodeBId: '21' },
  { id: '29', nodeAId: '17', nodeBId: '22' },
  { id: '30', nodeAId: '18', nodeBId: '20' },
  { id: '31', nodeAId: '18', nodeBId: '21' },
  { id: '32', nodeAId: '19', nodeBId: '20' },
  { id: '33', nodeAId: '20', nodeBId: '22' },
  { id: '34', nodeAId: '21', nodeBId: '22' },
  { id: '35', nodeAId: '22', nodeBId: '28' }, // right staircase

  // staircase left
  { id: '36', nodeAId: '23', nodeBId: '24' },
  { id: '37', nodeAId: '24', nodeBId: '25' },

  // staircase right
  { id: '38', nodeAId: '26', nodeBId: '27' },
  { id: '39', nodeAId: '27', nodeBId: '28' },
];

const graph: Graph = {
  id: '1',
  originX: 0,
  originY: 0,
  originZ: 0,
  scale: 1,
  nodes,
};

async function main() {
  // Create the graph first
  await prisma.graph.create({
    data: {
      id: graph.id,
      OriginX: graph.originX,
      OriginY: graph.originY,
      OriginZ: graph.originZ,
    },
  });

  // Create all nodes
  await Promise.all(
    graph.nodes.map((node) =>
      prisma.node.create({
        data: {
          id: node.id,
          x: node.x,
          y: node.y,
          z: node.z,
          label: node.label,
          graphId: node.graphId,
        },
      }),
    ),
  );

  // Create all edges
  await Promise.all(
    edges.map((edge) =>
      prisma.edge.create({
        data: {
          id: edge.id,
          nodeAId: edge.nodeAId,
          nodeBId: edge.nodeBId,
        },
      }),
    ),
  );

  // Create machines
  await prisma.machine.createMany({
    data: [
      {
        id: 1,
        name: 'Industrial Boiler',
        description:
          'High-capacity boiler system equipped with temperature and pressure monitoring.',
        nodeId: '1',
      },
      {
        id: 2,
        name: 'Hydraulic Press',
        description:
          'Precision hydraulic press with temperature, pressure, and vibration sensors for monitoring operational stress.',
        nodeId: '2',
      },
      {
        id: 3,
        name: 'Rotary Pump',
        description:
          'Compact rotary pump with vibration monitoring for predictive maintenance.',
        nodeId: '3',
      },
    ],
    skipDuplicates: true, // avoids error if re-run
  });
}

main()
  .catch((e) => {
    console.error(e);
    process.exit(1);
  })
  // eslint-disable-next-line @typescript-eslint/no-misused-promises
  .finally(() => prisma.$disconnect());
